/*
 * ssi_task
 *
 * Copyright (C) 2022 Texas Instruments Incorporated
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

/******************************************************************************
 *
 * vSSITaskCreate is called by main() to configure SSI0 as a master and SSI1
 * as a slave. vSSITaskCreate also creates two tasks: prvSSITransmitTask and
 * prvSSIReceiveTask. The transmit task sends four bytes to SSI0 and deletes
 * the task itself after four bytes are transmitted. The receive task waits for
 * the notification released from the xSSI1Handler ISR indicating the SSI1
 * receive FIFO is half full or more. Once the receive task is unblocked, it
 * reads and processes the data by means of displaying the received bytes to
 * the terminal window.
 *
 */

/* Standard includes. */
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <stdbool.h>

/* Kernel includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

/* Hardware includes. */
#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
#include "driverlib/gpio.h"
#include "driverlib/interrupt.h"
#include "driverlib/pin_map.h"
#include "driverlib/ssi.h"
#include "driverlib/sysctl.h"
#include "drivers/rtos_hw_drivers.h"
#include "utils/uartstdio.h"
/*-----------------------------------------------------------*/

/*
 * Number of bytes to send and receive.
 */
#define NUM_SSI_DATA            4

/*
 * Declare a variable that is used to hold the handle of the SSI1
 * interrupt task.
 */
TaskHandle_t xSSI1IntTask = NULL;

/*
 * The tasks as described in the comments at the top of this file.
 */
static void prvSSITransmitTask( void *pvParameters );
static void prvSSIReceiveTask( void *pvParameters );

/*
 * Called by main() to create the SSI tasks.
 */
void vSSITaskCreate( void );

/*
 * Configure the SSI peripheral for both Master and Slave operation.
 */
static void prvConfigureSSIMaster( void );
static void prvConfigureSSISlave( void );
/*-----------------------------------------------------------*/

void vSSITaskCreate( void )
{
    /* Configure the SSI0 peripheral for Master Mode at 2MHz. */
    prvConfigureSSIMaster();

    /* Configure the SSI1 peripheral for Slave Mode at 2MHz. */
    prvConfigureSSISlave();

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name Hello task - for debug only as it is
     *    not used by the kernel.
     *  - The size of the stack to allocate to the task.
     *  - No parameter passed to the task
     *  - The priority assigned to the task.
     *  - The task handle is NULL */
    xTaskCreate( prvSSITransmitTask,
                 "SSITX",
                 configMINIMAL_STACK_SIZE,
                 NULL,
                 tskIDLE_PRIORITY + 2,
                 NULL );

    /* Create the task as described in the comments at the top of this file.
     *
     * The xTaskCreate parameters in order are:
     *  - The function that implements the task.
     *  - The text name Hello task - for debug only as it is
     *    not used by the kernel.
     *  - The size of the stack to allocate to the task.
     *  - No parameter passed to the task
     *  - The priority assigned to the task.
     *  - The task handle is NULL */
    xTaskCreate( prvSSIReceiveTask,
                 "SSIRX",
                 configMINIMAL_STACK_SIZE,
                 NULL,
                 tskIDLE_PRIORITY + 1,
                 &xSSI1IntTask );
}
/*-----------------------------------------------------------*/

static void prvSSITransmitTask( void *pvParameters )
{
uint32_t pui32DataTx0[4] = {'T', 'I', 'V', 'A'};
uint32_t ui32Index;

    ui32Index = 0;

    UARTprintf("SSI0 Sending:\n  ");
    UARTprintf("'T' 'I' 'V' 'A' \n");

    for( ;; )
    {

        /* Dummy write to start slave which is required for Quad-SSI
         * mode operation. */
        SSIDataPut(SSI1_BASE, 0);

        /* Check if the last data byte is queued up to be sent. */
        if (ui32Index == (NUM_SSI_DATA - 1))
        {
            /* Calling SSIAdvDataPutFrameEnd on the last data will put out the
             * data and de-assert the the Fss pin. */
            SSIAdvDataPutFrameEnd(SSI0_BASE, pui32DataTx0[ui32Index]);
        }
        else
        {
            /* Send the data using the "blocking" put function.  This function
             * will wait until there is room in the send FIFO before returning.
             * This allows you to assure that all the data you send makes it into
             * the send FIFO. */
            SSIDataPut(SSI0_BASE, pui32DataTx0[ui32Index]);
        }

        /* Increment the index. */
        ui32Index++;

        if (ui32Index == 4)
        {
            /* Delete the task now that data has been sent. */
            vTaskDelete(NULL);
        }
    }
}
/*-----------------------------------------------------------*/

static void prvSSIReceiveTask( void *pvParameters )
{
unsigned long ulEventsToProcess;
uint32_t pui32DataRx1[4];
uint32_t ui32Index;

    for( ;; )
    {
        /* Wait to receive a notification sent directly to this task from the
         * interrupt service routine. */
        ulEventsToProcess = ulTaskNotifyTake( pdTRUE, portMAX_DELAY );

        if (ulEventsToProcess != 0)
        {
            /* There should be four bytes received. */
            for(ui32Index = 0; ui32Index < 4; ui32Index++)
            {
                /* Receive the data using the "blocking" Get function.  This
                 * function will wait until there is data in the receive FIFO
                 * before returning. */
                SSIDataGet(SSI1_BASE, &pui32DataRx1[ui32Index]);

                /* Since we are using 8-bit data, mask off the MSB. */
                pui32DataRx1[ui32Index] &= 0x00FF;
            }

            /* Display the data that SSI1 received. */
            UARTprintf("\nSSI1 Received:\n  ");
            UARTprintf("'%c' '%c' '%c' '%c'", pui32DataRx1[0],
                       pui32DataRx1[1], pui32DataRx1[2], pui32DataRx1[3]);

            /* Delete the task now that data has been printed. */
            vTaskDelete(NULL);
        }
    }
}
/*-----------------------------------------------------------*/

static void prvConfigureSSIMaster( void )
{
uint32_t ui32DummyRead;

    /* The SSI0 peripheral must be enabled for use. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_SSI0);

    /* This example uses Port A[7:2] for the SSI0 pins.  The GPIO port needs to
     * be enabled so those pins can be used. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);

    /* Configure the GPIO settings for the SSI pins.  This function also gives
    * control of these pins to the SSI hardware.  Consult the data sheet to
    * see which functions are allocated per pin. */

    /* Configure the pin muxing for SSI0 functions.  SSI0 uses PortA for the
     * SSICLK, SSIFss and the 4 data pins. GPIO ports need to be enabled so
     * those pins can be used.
     * The GPIO pins are assigned as follows:
     *   PA7 - SSI0XDAT3
     *   PA6 - SSI0XDAT2
     *   PA5 - SSI0XDAT1
     *   PA4 - SSI0XDAT0
     *   PA3 - SSI0Fss
     *   PA2 - SSI0CLK */
     GPIOPinConfigure(GPIO_PA2_SSI0CLK);
     GPIOPinConfigure(GPIO_PA3_SSI0FSS);
     GPIOPinConfigure(GPIO_PA4_SSI0XDAT0);
     GPIOPinConfigure(GPIO_PA5_SSI0XDAT1);
     GPIOPinConfigure(GPIO_PA6_SSI0XDAT2);
     GPIOPinConfigure(GPIO_PA7_SSI0XDAT3);

     GPIOPinTypeSSI(GPIO_PORTA_BASE, GPIO_PIN_7 | GPIO_PIN_6 | GPIO_PIN_5 |
                        GPIO_PIN_4 | GPIO_PIN_3 | GPIO_PIN_2);

    /* Configure and enable the SSI0 port for SPI master mode.  Use SSI0,
     * system clock supply, idle clock level low and active low clock in legacy
     * Freescale SPI mode, master mode, 2MHz SSI frequency, and 8-bit data. For
     * SPI mode, you can set the polarity of the SSI clock when the SSI unit is
     * idle.  You can also configure what clock edge you want to capture data
     * on.  Please reference the device datasheet for more information on the
     * different SPI modes. */
    SSIConfigSetExpClk(SSI0_BASE, configCPU_CLOCK_HZ, SSI_FRF_MOTO_MODE_0,
                       SSI_MODE_MASTER, 2000000, 8);

    /* Enable Quad-SSI mode for SSI0. */
    SSIAdvModeSet(SSI0_BASE, SSI_ADV_MODE_QUAD_WRITE);

    /* Hold the Fss pin low during transfers.  The Fss will be controlled
     * directly by the SSIAdvDataPutFrameEnd().  If calling
     * SSIAdvDataPutFrameEnd to write data to the FIFO, the Fss is de-asserted
     * for the corresponding data. */
    SSIAdvFrameHoldEnable(SSI0_BASE);

    /* Enable the SSI0 module. */
    SSIEnable(SSI0_BASE);

    /* Read any residual data from the SSI port.  This makes sure the receive
     * FIFOs are empty, so we don't read any unwanted junk.  This is done here
     * because the SPI SSI mode is full-duplex, which allows you to send and
     * receive at the same time.  The SSIDataGetNonBlocking function returns
     * "true" when data was returned, and "false" when no data was returned.
     * The "non-blocking" function checks if there is any data in the receive
     * FIFO and does not "hang" if there isn't. */
    while(SSIDataGetNonBlocking(SSI0_BASE, &ui32DummyRead))
    {
    }
}
/*-----------------------------------------------------------*/

static void prvConfigureSSISlave( void )
{
uint32_t ui32DummyRead;

    /* The SSI1 peripheral must be enabled for use. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_SSI1);

    /* This example uses Port B, D and E for the SSI1 pins.  The GPIO port
     * needs to be enabled so those pins can be used. */
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOB);
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOD);
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOE);

    /* Configure the GPIO settings for the SSI pins.  This function also gives
     * control of these pins to the SSI hardware.  Consult the data sheet to
     * see which functions are allocated per pin. */

    /* Configure the pin muxing for SSI1 functions.  SSI1 uses
     * PortB, PortD and PortE for the SSICLK, SSIFss and the 4 data pins.
     * GPIO ports need to be enabled so those pins can be used.
     * The GPIO pins are assigned as follows:
     *   PD5 - SSI1XDAT3
     *   PD4 - SSI1XDAT2
     *   PE5 - SSI1XDAT1
     *   PE4 - SSI1XDAT0
     *   PB4 - SSI1Fss
     *   PB5 - SSI1CLK */
    GPIOPinConfigure(GPIO_PB5_SSI1CLK);
    GPIOPinConfigure(GPIO_PB4_SSI1FSS);
    GPIOPinConfigure(GPIO_PE4_SSI1XDAT0);
    GPIOPinConfigure(GPIO_PE5_SSI1XDAT1);
    GPIOPinConfigure(GPIO_PD4_SSI1XDAT2);
    GPIOPinConfigure(GPIO_PD5_SSI1XDAT3);

    GPIOPinTypeSSI(GPIO_PORTB_BASE, GPIO_PIN_5 | GPIO_PIN_4 );
    GPIOPinTypeSSI(GPIO_PORTD_BASE, GPIO_PIN_5 | GPIO_PIN_4 );
    GPIOPinTypeSSI(GPIO_PORTE_BASE, GPIO_PIN_5 | GPIO_PIN_4 );

    /* Configure and enable the SSI1 port for SPI slave mode with matching
     * SPI mode, clock speed, and data size parameters as the master. */
    SSIConfigSetExpClk(SSI1_BASE, configCPU_CLOCK_HZ, SSI_FRF_MOTO_MODE_0,
                       SSI_MODE_SLAVE, 2000000, 8);

    /* Enable Quad-SSI mode for SSI1. */
    SSIAdvModeSet(SSI1_BASE, SSI_ADV_MODE_QUAD_READ);

    /* Enable SSI1 interrupt on RX FIFO half full or more. */
    SSIIntEnable(SSI1_BASE, SSI_RXFF);

    /* Enable the SSI1 interrupt in the NVIC. */
    IntEnable(INT_SSI1);

    /* Enable the SSI1 module. */
    SSIEnable(SSI1_BASE);

    /* Read any residual data from the SSI port.  This makes sure the receive
     * FIFOs are empty, so we don't read any unwanted junk.  This is done here
     * because the SPI SSI mode is full-duplex, which allows you to send and
     * receive at the same time.  The SSIDataGetNonBlocking function returns
     * "true" when data was returned, and "false" when no data was returned.
     * The "non-blocking" function checks if there is any data in the receive
     * FIFO and does not "hang" if there isn't. */
    while(SSIDataGetNonBlocking(SSI1_BASE, &ui32DummyRead))
    {
    }
}
/*-----------------------------------------------------------*/

void xSSI1Handler( void )
{
BaseType_t xHigherPriorityTaskWoken;
uint32_t ui32Status;

    /* Read SSIMIS (SSI Masked Interrupt Status). */
    ui32Status = SSIIntStatus(SSI1_BASE, true);

    /* Clear the SSI interrupt. */
    SSIIntClear(SSI1_BASE, ui32Status);

    /* Turn off the RX FIFO interrupt. */
    SSIIntDisable(SSI1_BASE, SSI_RXFF);

    /* The xHigherPriorityTaskWoken parameter must be initialized to pdFALSE as
     * it will get set to pdTRUE inside the interrupt safe API function if a
     * context switch is required. */
    xHigherPriorityTaskWoken = pdFALSE;

    /* Send a notification directly to the task to which interrupt processing
     * is being deferred. */
    vTaskNotifyGiveFromISR( xSSI1IntTask, &xHigherPriorityTaskWoken );

    /* This FreeRTOS API call will handle the context switch if it is required
     * or have no effect if that is not needed. */
    portYIELD_FROM_ISR( xHigherPriorityTaskWoken );
}
/*-----------------------------------------------------------*/

void vApplicationTickHook( void )
{
    /* This function will be called by each tick interrupt if
        configUSE_TICK_HOOK is set to 1 in FreeRTOSConfig.h.  User code can be
        added here, but the tick hook is called from an interrupt context, so
        code must not attempt to block, and only the interrupt safe FreeRTOS API
        functions can be used (those that end in FromISR()). */

    /* Only the full demo uses the tick hook so there is no code is
        executed here. */
}
